function [revenue, D, regret, relative_regret, beta_hat_vect, beta_error, price_dispersion] = Pricing_Strategy(T, C, S, pmin, pmax, h, true_beta, p1, p2, beta_hat_first, eps, delta)
% Simulates a pricing strategy over multiple selling seasons.
%
% Inputs:
%   T: Number of selling seasons.
%   C: Total capacity.
%   S: Number of time periods in each selling season.
%   pmin, pmax: Minimum and maximum prices.
%   h: Function handle for demand probability.
%   true_beta: True parameter vector for demand model.
%   p1, p2: Initial prices for the first and second demand realization.
%   beta_hat_first: Initial estimate for beta parameters.
%   eps: Tolerance for price dispersion comparison.
%   delta:  For a chosen delta the price that maximizes the revenue is between
%   pmin+delta and pmax-delta
%
% Outputs:
%   revenue: Accumulated revenue over time.
%   D: Demand realization matrix.
%   regret: Regret at the end of each selling season.
%   relative_regret: Relative regret at the end of each selling season.
%   beta_hat_vect: Estimated beta parameters over time.
%   beta_error: L2 norm of the difference between estimated and true beta.
%   price_dispersion: Minimum eigenvalue of the price dispersion matrix.


if length(beta_hat_first) == 2
    len = 2;
else
    len = 3;
end


%INITIALIZATIONS

% regret and relative regret vectors
regret = zeros(1,T);
relative_regret = zeros(1,T);

% beta estimates vector
beta_hat = beta_hat_first;
beta_hat_vect = zeros(len,T*S-1);
beta_hat_vect(:,1) = beta_hat;

% beta estimates errors vector
beta_error = zeros(1,T*S-1);
beta_error(1) = vecnorm(beta_hat - true_beta, 2);

% price dispersion vector
price_dispersion = zeros(1,T*S);

if len==2
    P = [1, p1(1,1); p1(1,1), p1(1,1)^2];
    price_dispersion(1) = min(eig(P));
    P = P + [1, p2(1,1); p2(1,1), p2(1,1)^2];
    price_dispersion(2) = min(eig(P));
else
    P = [1, p1(1,1), S; p1(1,1), p1(1,1)^2, p1(1,1)*S; S, p1(1,1)*S, S^2];
    price_dispersion(1) = min(eig(P));
    P = P + [1, p2(1,1), S-1; p2(1,1), p2(1,1)^2, p2(1,1)*(S-1); S-1, p2(1,1)*(S-1), (S-1)^2];
    price_dispersion(2) = min(eig(P));
end

% demand vector
D = zeros(len,T*S);
D(:,1)= p1; % the first price and first demand realization

c = (C+1)-D(2,1); % c goes from 0 to C
revenue = D(1,1)*D(2,1);

D(:,2)= p2; % the second price



[~, true_value] = Optimization_Phase(C, S, pmin, pmax, h, true_beta);

flag_break = 0;

SS = 1; % first selling season
fprintf('Selling season %d\n', SS)

for t = 2:T*S-1 % for all time periods over all selling seasons
    
    if c == 1 % in the case of initial inventory 1 and product sold in the first time period
        flag_break = 1;
        c = C+1; % restore capacity
        regret(SS) = SS*true_value-revenue;
        relative_regret(SS) = (regret(SS)/(SS*true_value))*100;
        SS = SS + 1; % go to the next selling season
        if SS ~= T + 1
            fprintf('Selling season %d\n', SS)
            price_dispersion_next = price_dispersion(t);
            price_dispersion(t) = price_dispersion(t-1);
            D(1,1 + (SS-1)*S) = D(1,t);
        end
        D(1,t) = 0;
    end
    
    if flag_break == 1 && t ~= 1 + (SS-1)*S
        price_dispersion(t+1) = price_dispersion(t);
        beta_error(t) = beta_error(t-1);
        continue;
    elseif flag_break == 1 && t == 1 + (SS-1)*S % if we are at the start of the next selling season
        beta_error(t) = beta_error(t-1);
        price_dispersion(t) = price_dispersion_next;
        flag_break = 0;
    end
    
    if len == 2
        D(2,t) = binornd(1, h(true_beta(1)+true_beta(2)*D(1,t))); % realization of the demand at time t
    else
        if t == SS*S
            DTD = 1;
        else
            DTD = S - mod(t, S) + 1; % number of days to departure
        end
        D(2,t) = binornd(1, h(true_beta(1)+true_beta(2)*D(1,t)+true_beta(3)*DTD)); % realization of the demand at time t
        D(3,t) = DTD;
        
    end
    revenue = revenue + D(1,t)*D(2,t); % collection of revenue
    
    if D(2,t) == 1
        c = c-1; % the remaining capacity at time t+1
    end
    
    D_data = D(:,1:t);
    D_data( :, all(~D_data,1) )= []; % we eliminate time periods in which no sale was made
    
    beta_hat = Estimation_Phase(D_data, h, beta_hat, pmin, pmax, delta, S); % new parameters estimate
    beta_hat_vect(:,t) = beta_hat;
    beta_error(t) = vecnorm(beta_hat - true_beta, 2);
    
    price_matrix = Optimization_Phase(C, S, pmin, pmax, h, beta_hat);
    st = mod(t,S);
    if t == S*SS || c == 1 
        popt = price_matrix(C+1, 1);
    else
        popt = price_matrix(c, st+1);
    end
    
    
    flag1 = 0;
    flag2 = 0;
    
    % check if we are at the end of the selling season or only one seat is left
    if (t+1 == S*SS || c == 2) && (D(1,1 + (SS-1)*S) ~= 0) % if we are at the beginning of the selling season we don't have prices to compare
        flag1 = 1;
        flag2 = 1;
        
        % check if the prices used in the selling season are too close to the new one
        for i = 1 + (SS-1)*S : t
            if abs(D(1,i)-popt) > eps
                flag2 = 0;
                break;
            end
            % check if the prices used in the selling season are too close to eachother
            for j = 1 + (SS-1)*S + 1 : t
                if abs(D(1,i)-D(1,j)) > eps
                    flag1 = 0;
                    break;
                end
            end
            if flag1 == 0
                break;
            end
        end
    end
    
    if flag1 == 1 && flag2 == 1
        
        % Set the bounds for the next price
        lower_bound = popt - 2*eps;
        upper_bound = popt + 2*eps;
        
        if (upper_bound >= pmin) && (upper_bound <= pmax) && ((lower_bound < pmin) || (lower_bound > pmax))
            D(1,t+1) = upper_bound;
            
        elseif (lower_bound >= pmin) && (lower_bound <= pmax) && ((upper_bound < pmin) || (upper_bound > pmax))
            D(1,t+1) = lower_bound;
            
        elseif (lower_bound >= pmin) && (lower_bound <= pmax) && (upper_bound >= pmin) && (upper_bound <= pmax)
            prices = [lower_bound, upper_bound];
            r = randi([1, 2], 1); % Get a 1 or 2 randomly.
            D(1,t+1) = prices(r);  % Extract the price for this random number.
        end
    else
        D(1,t+1) = popt;
    end
    
    
    % Calculate the price dispersion at time period t+1
    if len==2
        P = P + [1, D(1,t+1); D(1,t+1), D(1,t+1)^2];
    else
        P = P + [1, D(1,t+1), D(3,t)-1; D(1,t+1), D(1,t+1)^2, D(1,t+1)*(D(3,t)-1); D(3,t)-1, D(1,t+1)*(D(3,t)-1), (D(3,t)-1)^2];
        
    end
    price_dispersion(t+1) = min(eig(P));
    
    if t == T*S-1 && c > 1 % if there's capacity left
        if len == 2
            D(2,t+1) = binornd(1, h(true_beta(1)+true_beta(2)*D(1,t+1)));
        else
            D(2,t+1) = binornd(1, h(true_beta(1)+true_beta(2)*D(1,t+1)+true_beta(3)));
            D(3, t) = S-1;
            D(3,t+1) = mod(t,S) + 1;
        end
        revenue = revenue + D(1,t+1)*D(2,t+1);
    elseif t == T*S-1 && c == 1
        D(1,t+1) = 0;
    end
    
    if t == SS*S || t == T*S-1 % we are at the end of the selling season or at the end of the time periods
        regret(SS) = SS*true_value-revenue;
        relative_regret(SS) = (regret(SS)/(SS*true_value))*100;
        SS = SS+1;
        fprintf('Selling season %d\n', SS)
        if SS == T + 1
            break;
        end
        c = C+1;
    end
    
end


end

